/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package burp;

import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 * @author t.isayama
 */
public class BurpExtender implements IBurpExtender, IHttpListener {

    public enum HighlightColor {
        WHITE, RED, ORANGE, YELLOW, GREEN, CYAN, BLUE, PINK, MAGENTA, GRAY;

        @Override
        public String toString() {
            return name().toLowerCase();
        }

    };

    private IBurpExtenderCallbacks callbacks = null;
    private final Map<Pattern, HighlightColor> bannerMap = new HashMap();
    private boolean burpProfessional = false;

    @Override
    /* IBurpExtender interface implements method */
    public void registerExtenderCallbacks(IBurpExtenderCallbacks callbacks) {
        this.callbacks = callbacks;

        // burp Professional 判定
        String[] version = callbacks.getBurpVersion();
        this.burpProfessional = (0 <= version[0].indexOf("Professional"));

        // banner登録
        this.bannerMap.put(Pattern.compile("(Apache/[0-9.]+)"), HighlightColor.GREEN);
        this.bannerMap.put(Pattern.compile("(PHP/[0-9.]+)"), HighlightColor.GREEN);
        // 必要に応じて追加

        // プロフェッショナル版の場合
        if (this.burpProfessional) {
            callbacks.registerScannerCheck(professionalPassiveScanCheck());
        } // フリー版の場合
        else {
            callbacks.registerHttpListener(this);
        }

    }

    // プロフェッショナル版の実装
    public IScannerCheck professionalPassiveScanCheck() {
        return new IScannerCheck() {
            @Override
            public List<IScanIssue> doPassiveScan(IHttpRequestResponse baseRequestResponse) {
                String message = getByteToStr(baseRequestResponse.getResponse());
                List<int[]> responseMarkers = new ArrayList<>();
                for (Pattern ptn : bannerMap.keySet()) {
                    Matcher m = ptn.matcher(message);
                    while (m.find()) {
                        // 検出範囲の追加
                        responseMarkers.add(new int[]{m.start(), m.end()});
                    }
                }
                if (responseMarkers.size() > 0) {
                    List<IScanIssue> issues = new ArrayList<>();
                    IHttpRequestResponseWithMarkers messageInfoMark = callbacks.applyMarkers(baseRequestResponse, null, responseMarkers);
                    issues.add(makeBannerIssue(messageInfoMark));
                    return issues;
                }
                return null;
            }

            @Override
            public List<IScanIssue> doActiveScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
                return null;
            }

            @Override
            public int consolidateDuplicateIssues(IScanIssue existingIssue, IScanIssue newIssue) {
                if (existingIssue.getIssueName().equals(newIssue.getIssueName())) {
                    // 同一とみなせる場合は報告をスキップ
                    return -1;
                }
                return 0;
            }

        };
    }

    public IScanIssue makeBannerIssue(final IHttpRequestResponse messageInfo) {
        return new IScanIssue() {
            @Override
            public URL getUrl() {
                IRequestInfo reqInfo = callbacks.getHelpers().analyzeRequest(messageInfo.getHttpService(), messageInfo.getRequest());
                return reqInfo.getUrl();
            }

            @Override
            public String getIssueName() {
                return "Server Banner Disclosure";
            }

            @Override
            public int getIssueType() {
                /**
                 * https://portswigger.net/knowledgebase/issues/ Extension generated issue
                 */
                return 0x08000000;
            }

            @Override
            public String getSeverity() {
                return "Information";
            }

            @Override
            public String getConfidence() {
                return "Certain";
            }

            @Override
            public String getIssueBackground() {
                return "omission";
            }

            @Override
            public String getRemediationBackground() {
                return "omission";
            }

            @Override
            public String getIssueDetail() {
                return "omission";
            }

            @Override
            public String getRemediationDetail() {
                return "omission";
            }

            @Override
            public IHttpRequestResponse[] getHttpMessages() {
                return new IHttpRequestResponse[]{messageInfo};
            }

            @Override
            public IHttpService getHttpService() {
                return messageInfo.getHttpService();
            }
        };
    }

    // フリー版の実装
    @Override
    public void processHttpMessage(int toolFlag, boolean messageIsRequest, IHttpRequestResponse messageInfo) {
        if (toolFlag == IBurpExtenderCallbacks.TOOL_PROXY && !messageIsRequest) {
            freePassiveScan(messageInfo);
        }
    }

    public void freePassiveScan(IHttpRequestResponse messageInfo) {
        String message = getByteToStr(messageInfo.getResponse());
        for (Pattern ptn : this.bannerMap.keySet()) {
            Matcher m = ptn.matcher(message);
            if (m.find()) {
                String bunner = m.group(1);
                HighlightColor color = this.bannerMap.get(ptn);
                // 既にコメントがある場合は追記                       
                String comment = (messageInfo.getComment() == null) ? "" : messageInfo.getComment() + " ";
                messageInfo.setComment(comment + bunner);
                messageInfo.setHighlight(color.toString());
            }
        }
    }

    // 共通処理
    // 文字列を生のバイト列に変換
    public static byte[] getStrToByte(String str) {
        return str.getBytes(StandardCharsets.ISO_8859_1);
    }

    // 生のバイト列を文字列に変換
    public static String getByteToStr(byte[] bytes) {
        return new String(bytes, 0, bytes.length, StandardCharsets.ISO_8859_1);
    }

}
